﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Collections.Specialized;
using System.IO;
using System.Xml.Serialization;
using System.Configuration;

namespace PerformanceTesting
{
    public static class Utils
    {
        public static string ToUriData(this NameValueCollection collection)
        {
            string result = GetData(collection);
            if (!string.IsNullOrEmpty(result))
                return "?" + result;
            return result;
        }

        public static string ToPostData(this NameValueCollection collection)
        {
            return GetData(collection);
        }

        public static NameValueCollection ToNameValueCollection(this List<Parameter> parameters)
        {
            NameValueCollection collection = new NameValueCollection(parameters.Count);
            foreach (Parameter p in parameters)
                collection.Add(p.Key, p.Value);
            return collection;
        }

        private static string GetData(NameValueCollection collection)
        {
            string result = string.Empty;
            if (collection != null && collection.Count > 0)
            {
                foreach (string key in collection.AllKeys)
                    result += key + "=" + Uri.EscapeDataString(collection[key]) + "&";
                result = result.Substring(0, result.Length - 1);
            }
            return result;
        }

        public static bool IsLoggedOn(string responseText)
        {
            if (string.IsNullOrEmpty(responseText) ||
               responseText.IndexOf(@"<title>BMS User Login Page</title>", StringComparison.InvariantCultureIgnoreCase) >= 0 ||
               responseText.IndexOf(@"<h2>Object moved to <a href=""/Exception/Index"">", StringComparison.InvariantCultureIgnoreCase) >= 0)
                return false;
            return true;
        }

        public static bool IsNotException(string responseText)
        {
            if (!IsLoggedOn(responseText))
                return false;
            if (responseText.IndexOf(@"<title>Unhandled Exception</title>", StringComparison.InvariantCultureIgnoreCase) >= 0 ||
                responseText.IndexOf(@"<h2>Object moved to <a href=""/Exception/Index"">", StringComparison.InvariantCultureIgnoreCase) >=0)
                return false;
            return true;
        }

        public static void PrintDataToConsole()
        {
            string format = GetFormat();
            Console.WriteLine(format, Constants.ActionColumnName, Constants.AverageColumnName, Constants.CallsColumnName);
            foreach (string action in Logger.LogData.Keys)
            {
                long average = (long)Logger.LogData[action].Average(log => log.Duration);
                Console.WriteLine(format, action, average, Logger.LogData[action].Count);
            }

            if (Logger.Exceptions.Count > 0)
                Console.WriteLine("{0} exceptions occured durring the test.");
        }

        public static void PrintDataToOutputFile(string outputFilePath, string responseFilePath, string errorFilePath, string exceptionFilePath, bool deleteOutputFile)
        {
            try
            {
                File.Delete(responseFilePath);
            }
            catch { }
            try
            {
                File.Delete(errorFilePath);
            }
            catch { }
            try
            {
                if (deleteOutputFile)
                    File.Delete(outputFilePath);
            }
            catch { }

            string separator = new string('*', 250);
            string format = GetFormat();
            TextWriter writer = new StreamWriter(outputFilePath, !deleteOutputFile);
            TextWriter responseWriter = new StreamWriter(responseFilePath, false);
            writer.WriteLine(DateTime.Now.ToShortDateString() + " " + DateTime.Now.ToShortTimeString());
            writer.WriteLine(format, Constants.ActionColumnName, Constants.AverageColumnName, Constants.CallsColumnName);
            foreach (string action in Logger.LogData.Keys)
            {
                long average = (long)Logger.LogData[action].Average(log => log.Duration);
                writer.WriteLine(format, action, average, Logger.LogData[action].Count);
                foreach(Log log in Logger.LogData[action])
                { 
                    responseWriter.WriteLine(log.Details);
                    responseWriter.WriteLine(separator);
                }
            }

            writer.WriteLine("Detailed view");
            foreach (string action in Logger.LogData.Keys) 
            {
                writer.WriteLine(action);
                string detailedDurations = string.Join(",", Logger.LogData[action].Select<Log, string>(log => log.Duration + ""));
                writer.WriteLine(detailedDurations);
                writer.WriteLine(separator);
            }

            writer.WriteLine(new string('=', 250));
            writer.Flush();
            writer.Close();

            if (Logger.Errors.Count > 0)
                PrintErrorsToFile(errorFilePath);
            if(Logger.Exceptions.Count > 0)
                PrintExceptionsToFile(exceptionFilePath);
        }

        public static int Timeout
        {
            get
            {
                return Int32.Parse(ConfigurationManager.AppSettings[Constants.MaxThreadNumber]) * 5 * 1000;
            }
        }

        private static void PrintErrorsToFile(string path) 
        {
            string separator = new string('*', 250);
            TextWriter writer = new StreamWriter(path, false);

            foreach (string key in Logger.Errors.Keys)
            {
                writer.WriteLine(key);
                writer.WriteLine(Logger.Errors[key]);
                writer.WriteLine(separator);
            }

            writer.Flush();
            writer.Close();
        }

        private static void PrintExceptionsToFile(string path)
        {
            XmlSerializer serializer = new XmlSerializer(typeof(SerializableException));

            FileStream stream = new FileStream(path, FileMode.Create);

            foreach (Exception e in Logger.Exceptions)
            {
                SerializableException se = new SerializableException(e);
                serializer.Serialize(stream, se);
            }

            stream.Flush();
            stream.Close();
        }

        private static string GetFormat()
        {
            if (Logger.LogData.Count == 0)
                return string.Empty;

            int maxActionLength = Logger.LogData.Keys.Max(action => action.Length);
            int maxNumberLength = (Logger.LogData.Values.Max(list => list.Max(log => log.Duration)) + "").Length;
            int maxCountLength = (Logger.LogData.Keys.Max(action => Logger.LogData[action].Count) + "").Length;

            if (maxActionLength < Constants.ActionColumnName.Length)
                maxActionLength = Constants.ActionColumnName.Length;
            if (maxNumberLength < Constants.AverageColumnName.Length)
                maxNumberLength = Constants.AverageColumnName.Length;
            if (maxCountLength < Constants.CallsColumnName.Length)
                maxCountLength = Constants.CallsColumnName.Length;

            maxActionLength += 5;
            maxCountLength += 2;

            string format = "{0, -" + maxActionLength + "}{1, " + maxNumberLength + "}{2, " + maxCountLength + "}";
            return format;
        }
    }

    public delegate void WaitCallback<T>(T state);

    public static class ThreadPoolHelper
    {

        public static bool QueueUserWorkItem<T>(T state, WaitCallback<T> callback)
        {
            return System.Threading.ThreadPool.QueueUserWorkItem(s => callback((T)s), state);
        }
    }
}
